document.addEventListener("DOMContentLoaded", function () {
  var tabContainer = document.querySelector("#ib-tab");
  var contentsContainer = document.querySelector("#ib-contents");

  // ✅ IBTab 인스턴스 생성
  createIBTab(tabContainer, contentsContainer, "tabManager", {
    widthTabBar: "100%",
    widthContents: "100%",
    heightContents: "calc(100vh - 50px)",
  });

  // ✅ 초기 HOME 탭 추가
  setTimeout(() => initializeTabs(), 200);

  // ✅ HOME 탭의 닫기 버튼 감시 (MutationObserver 활용)
  observeHomeTabCloseButton();
});

// ✅ HOME 탭의 닫기 버튼을 자동으로 감지하여 삭제하는 함수
function observeHomeTabCloseButton() {
  const observer = new MutationObserver(() => {
    const homeTab = document.querySelector(`#ib-tab [data-id="dashboard/dashboard"]`);
    if (homeTab) {
      const closeButton = homeTab.querySelector(".ib-tab-close");
      if (closeButton) {
        closeButton.remove(); // ✅ HOME 탭의 닫기 버튼을 강제로 삭제
        console.log("✅ HOME 탭 닫기 버튼 제거 완료");
      }
    }
  });

  observer.observe(document.querySelector("#ib-tab"), {
    childList: true,
    subtree: true,
  });
}

// ✅ HOME 탭 추가 함수 (초기 실행)
async function initializeTabs() {
  const content = await getTabContent("dashboard/dashboard");
  var homeTab = tabManager.findTabId("dashboard/dashboard");

  if (!homeTab) {
    tabManager.addTab({
      tabs: {
        title: '<i class="icon-home"></i> HOME',
        id: "dashboard-home",
        btnClose: false, // ✅ 기본적으로 닫기 버튼 비활성화 (무시될 수 있음)
        focus: true,
        className: "home-tab",
      },
      contents: { contents: content },
    });

    console.log("✅ HOME 탭이 추가됨");
  }
}

// ✅ 탭 제목 설정 함수
function getTabTitle(tabId) {
  const titles = {
    "dashboard/dashboard": "HOME",
    "franchiseInfo/franchiseCheck": "가맹점 조회",
    "franchiseInfo/franchiseRegister": "가맹점 신규등록",
    "franchiseInfo/commissionCheck": "수수료 정보 조회",
  };

  return titles[tabId] || "새로운 탭";
}

// ✅ HTML 파일 동적 로드 함수
async function getTabContent(tabId) {
  let filePath = `/html/pages/${tabId.replace("-", "/")}.html`;

  try {
    const response = await fetch(filePath);
    if (!response.ok) throw new Error(`❌ 페이지를 찾을 수 없습니다: ${filePath}`);
    return await response.text();
  } catch (error) {
    console.error(error);
    return `<div class="sub-content">🚨 콘텐츠를 불러올 수 없습니다.</div>`;
  }
}

// ✅ 탭 열기 및 이동 (이미 존재하면 이동)
async function openTab(tabId) {
  if (tabId === "dashboard/dashboard") {
    tabManager.goToTab(0);
    return;
  }

  var tabTitle = getTabTitle(tabId);
  var existingTab = tabManager.findTabId(tabId);

  if (existingTab) {
    tabManager.goToTab(existingTab.getIndex()); // ✅ 이미 존재하면 해당 탭으로 이동
    return;
  }

  const content = await getTabContent(tabId);

  tabManager.addTab({
    tabs: {
      title: tabTitle,
      id: tabId,
      btnClose: true, // ✅ HOME이 아닌 경우 닫기 버튼 활성화
      focus: true,
    },
    contents: { contents: content },
  });
  console.log(`✅ ${tabTitle} (${tabId}) 탭이 추가됨!`);
}

// ✅ 사이드바 클릭 시 해당 탭 열기
document.querySelectorAll(".sidebar-link").forEach((item) => {
  item.addEventListener("click", function () {
    var tabId = this.getAttribute("data-tab");
    openTab(tabId);
  });
});

// ✅ 탭 삭제 오류 방지
function closeTab(tabId) {
  var tab = tabManager.findTabId(tabId);

  if (!tab) {
    console.warn(`🚨 탭이 존재하지 않습니다: ${tabId}`);
    return;
  }

  try {
    tabManager.removeTab(tab.getIndex()); // ✅ 존재하는 경우만 삭제
  } catch (error) {
    console.error("❌ 탭 삭제 중 오류 발생:", error);
  }
}

// ✅ 탭 제거할 때 부모가 있는지 체크 후 삭제
function removeTabElement(tabId) {
  var tabElement = document.getElementById(tabId);
  if (tabElement && tabElement.parentNode) {
    tabElement.parentNode.removeChild(tabElement);
  } else {
    console.warn(`🚨 제거할 탭이 없습니다: ${tabId}`);
  }
}

// ✅ 페이지 로드 시 HOME 탭 추가
window.onload = function () {
  setTimeout(() => initializeTabs(), 100);
  setTimeout(() => {
    ibChart.create(); // ✅ 차트는 탭이 로드된 후 실행
  }, 200); // ✅ 차트 렌더링 지연
};
